/*
 * Copyright European Commission's
 * Taxation and Customs Union Directorate-General (DG TAXUD).
 */
package eu.europa.ec.taxud.cesop.validation;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.util.Collections;
import java.util.Iterator;

import eu.europa.ec.taxud.cesop.domain.MessageTypeEnum;
import eu.europa.ec.taxud.cesop.domain.MessageTypeIndicEnum;
import eu.europa.ec.taxud.cesop.domain.ValidationError;
import eu.europa.ec.taxud.cesop.domain.XmlMessageSpec;
import eu.europa.ec.taxud.cesop.domain.XmlPaymentDataMsg;
import eu.europa.ec.taxud.cesop.domain.XmlPaymentDataMsgPart;
import eu.europa.ec.taxud.cesop.readers.IPspXmlReader;
import eu.europa.ec.taxud.cesop.readers.PspXmlReader;
import eu.europa.ec.taxud.cesop.xsd.XsdSchema;

/**
 * Default implementation of {@link IPspFileValidator}
 */
public class PspFileValidator implements IPspFileValidator {

    private static final String GENERIC_MS = "BE";
    private static final String UUID_V4_ZERO = "00000000-0000-4000-8000-000000000000";

    /**
     * The Number transactions by psp part.
     */
    final int numberTransactionsByPspPart;
    /**
     * The default country for VLD generation
     */
    final String defaultCountry;
    /**
     * The Validation settings.
     */
    final ValidationSettings validationSettings;

    /**
     * Instantiates a new Psp file validator.
     *
     * @param numberTransactionsByPspPart the number transactions by psp part
     * @param validationSettings          the validation settings
     * @param defaultCountry              the default country for VLD generation
     */
    public PspFileValidator(int numberTransactionsByPspPart, ValidationSettings validationSettings, String defaultCountry) {
        this.numberTransactionsByPspPart = numberTransactionsByPspPart;
        this.validationSettings = validationSettings;
        this.defaultCountry = defaultCountry;
    }

    /**
     * Create default psp file validator.
     *
     * @return the psp file validator
     */
    public static PspFileValidator createDefault() {
        final int numberTransactionsByPspPart = 1000; // only relevant for throughput optimization
        return new PspFileValidator(numberTransactionsByPspPart, ValidationSettings.createDefault(), GENERIC_MS);
    }

    @Override
    public PspValidationResult validate(final File pspFile) {
        if (!pspFile.exists() || !pspFile.isFile()) {
            throw new CesopValidationException("The provided path is not a file: " + pspFile.getAbsolutePath());
        }

        try (final InputStream inputStream = Files.newInputStream(pspFile.toPath())) {
            return validate(inputStream, pspFile.length());
        } catch (final IOException e) {
            // bad input file, thus "expected" exception
            throw new CesopValidationException(e.getMessage());
        }
    }

    /**
     * Validate psp validation result.
     *
     * @param inputStream   the input stream
     * @param contentLength the content length
     * @return the psp validation result
     */
    public PspValidationResult validate(final InputStream inputStream, final long contentLength) {
        String messageRefId = null;
        XmlMessageSpec xmlMessageSpec = null;
        PspValidatingReader validationReader = null;
        MessageTypeEnum messageType = null;
        try (final IPspXmlReader pspXmlReader = new PspXmlReader(inputStream, contentLength, numberTransactionsByPspPart, true)) {
            xmlMessageSpec = pspXmlReader.getXmlMessageSpec();
            messageRefId = xmlMessageSpec.getMessageRefId();
            messageType = xmlMessageSpec.getMessageType();
            validationReader = new PspValidatingReader(pspXmlReader, validationSettings,
                    messageType.isBusinessRulesCheckEnabled(), null);

            if (messageType.isBusinessRulesCheckEnabled()) {
                if (validationReader.getValidationErrors().isEmpty()) {
                    Iterator<XmlPaymentDataMsgPart> iterator = validationReader.validateParts();
                    while (iterator.hasNext()) {
                        iterator.next(); // just go through to collect errors
                    }
                }
            }

            return new PspValidationResult(createPaymentDataMsg(xmlMessageSpec), validationReader.getValidationErrors(),
                    messageType);
        } catch (final Exception e) {
            if (validationReader != null) {
                if (e instanceof CesopTooManyErrorsException) {
                    return new PspValidationResult(createPaymentDataMsg(xmlMessageSpec),
                            validationReader.getValidationErrors(), messageType);

                } else {
                    final ValidationError validationError = ExceptionHandler.handleException(e, messageRefId);
                    validationReader.getValidationErrors().add(validationError);
                    return new PspValidationResult(createPaymentDataMsg(xmlMessageSpec),
                            validationReader.getValidationErrors(), messageType);
                }
            }
            return new PspValidationResult(createPaymentDataMsg(xmlMessageSpec),
                    Collections.singletonList(ExceptionHandler.handleException(e, messageRefId)), messageType);
        }
    }

    private XmlPaymentDataMsg createPaymentDataMsg(final XmlMessageSpec xmlMessageSpec) {
        final XmlPaymentDataMsg paymentDataMsg = new XmlPaymentDataMsg();
        if (xmlMessageSpec == null) {
            paymentDataMsg.setCesopVersion(XsdSchema.CURRENT_VERSION);
            paymentDataMsg.setCountry(defaultCountry);
            paymentDataMsg.setMessageRefId(UUID_V4_ZERO);
            paymentDataMsg.setMessageType(MessageTypeIndicEnum.CESOP100.getLabel());
            paymentDataMsg.setReportingPeriod(this.validationSettings.getBusinessMinPeriod());
        } else {
            paymentDataMsg.setCesopVersion(xmlMessageSpec.getXsdVersion());
            paymentDataMsg.setCountry(xmlMessageSpec.getTransmittingCountry());
            paymentDataMsg.setMessageType(xmlMessageSpec.getMessageTypeIndic().getLabel());
            paymentDataMsg.setMessageRefId(xmlMessageSpec.getMessageRefId());
            paymentDataMsg.setReportingPeriod(xmlMessageSpec.getReportingPeriod());
            if (xmlMessageSpec.getSendingPsp() != null) {
                paymentDataMsg.setSendingPspIdentifier(xmlMessageSpec.getSendingPsp().getPspId().getValue());
                paymentDataMsg.setSendingPspIdentifierType(xmlMessageSpec.getSendingPsp().getPspId().getType());
                paymentDataMsg.setSendingPspIdentifierOther(xmlMessageSpec.getSendingPsp().getPspId().getOther());
                paymentDataMsg.setSendingPspNames(xmlMessageSpec.getSendingPsp().getNames());
            }
        }

        return paymentDataMsg;
    }
}
